<?php
session_start();
require_once '../config/database.php';

if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] != 'admin') {
    header('Location: ../index.php');
    exit();
}

$admin_id = $_SESSION['user_id'];
$subject_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if (!$subject_id) {
    header('Location: subjects.php');
    exit();
}

$conn = getDBConnection();

// Get user data including profile picture
$user_query = "SELECT profile_picture FROM users WHERE id = ?";
$stmt = $conn->prepare($user_query);
$stmt->bind_param("i", $admin_id);
$stmt->execute();
$user_result = $stmt->get_result();
$user = $user_result->fetch_assoc();
$stmt->close();

// Get subject details
$subject_query = "SELECT s.*, u.full_name as instructor_name, u.email as instructor_email
                  FROM subjects s
                  LEFT JOIN users u ON s.instructor_id = u.id
                  WHERE s.id = ?";
$stmt = $conn->prepare($subject_query);
$stmt->bind_param("i", $subject_id);
$stmt->execute();
$subject_result = $stmt->get_result();
$subject = $subject_result->fetch_assoc();
$stmt->close();

if (!$subject) {
    header('Location: subjects.php');
    exit();
}

// Get enrolled students
$students_query = "SELECT u.*, e.enrolled_at
                   FROM users u
                   JOIN enrollments e ON u.id = e.student_id
                   WHERE e.subject_id = ?
                   ORDER BY u.year_level, u.section, u.full_name";
$stmt = $conn->prepare($students_query);
$stmt->bind_param("i", $subject_id);
$stmt->execute();
$students_result = $stmt->get_result();
$students = [];
while ($row = $students_result->fetch_assoc()) {
    $students[] = $row;
}
$stmt->close();

// Get activities count
$activities_query = "SELECT COUNT(*) as count FROM activities WHERE subject_id = ?";
$stmt = $conn->prepare($activities_query);
$stmt->bind_param("i", $subject_id);
$stmt->execute();
$activities_count = $stmt->get_result()->fetch_assoc()['count'];
$stmt->close();

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($subject['subject_name']); ?> - CODELAB</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <script src="../assets/js/dark-mode.js" defer></script>
</head>
<body>
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <?php if (isset($user['profile_picture']) && $user['profile_picture']): ?>
                    <img src="../<?php echo htmlspecialchars($user['profile_picture']); ?>" alt="Profile" class="nav-profile-pic">
                <?php else: ?>
                    <div class="nav-avatar">👨‍💼</div>
                <?php endif; ?>
                <h2>CODELAB ADMIN</h2>
            </div>
            <button class="burger-menu" onclick="toggleMenu()">
                <span></span>
                <span></span>
                <span></span>
            </button>
            <div class="nav-menu" id="nav-menu">
                <a href="dashboard.php" class="nav-link">Dashboard</a>
                <a href="users.php" class="nav-link">Manage Users</a>
                <a href="sections.php" class="nav-link">Students by Section</a>
                <a href="subjects.php" class="nav-link active">Manage Subjects</a>
                <a href="activities.php" class="nav-link">Manage Activities</a>
                <a href="profile.php" class="nav-link">Profile</a>
                <a href="../auth/logout.php" class="nav-link logout">Logout</a>
            </div>
        </div>
    </nav>

    <div class="main-content">
        <div class="dashboard-header">
            <div>
                <a href="subjects.php" style="color: var(--primary-moss); text-decoration: none;">← Back to Subjects</a>
                <h1>📚 <?php echo htmlspecialchars($subject['subject_name']); ?></h1>
                <p><?php echo htmlspecialchars($subject['subject_code']); ?></p>
            </div>
        </div>

        <!-- Subject Info -->
        <div class="dashboard-section full-width">
            <div class="section-header">
                <h2>📋 Subject Information</h2>
            </div>
            <div class="info-grid">
                <div class="info-item">
                    <strong>Description:</strong>
                    <p><?php echo htmlspecialchars($subject['description']); ?></p>
                </div>
                <div class="info-item">
                    <strong>Instructor:</strong>
                    <p>👨‍🏫 <?php echo htmlspecialchars($subject['instructor_name'] ?? 'No Instructor'); ?>
                    <?php if ($subject['instructor_email']): ?>
                        <br><small style="color: var(--text-secondary);">📧 <?php echo htmlspecialchars($subject['instructor_email']); ?></small>
                    <?php endif; ?>
                    </p>
                </div>
                <div class="info-item">
                    <strong>Statistics:</strong>
                    <p>👥 <?php echo count($students); ?> Students | 📝 <?php echo $activities_count; ?> Activities</p>
                </div>
            </div>
        </div>

        <!-- Enrolled Students -->
        <div class="dashboard-section full-width">
            <div class="section-header">
                <h2>👥 Enrolled Students (<?php echo count($students); ?>)</h2>
            </div>
            <?php if (!empty($students)): ?>
                <div class="students-grid">
                    <?php foreach ($students as $student): ?>
                        <div class="student-card">
                            <div class="student-avatar-container">
                                <?php if (isset($student['profile_picture']) && $student['profile_picture']): ?>
                                    <img src="../<?php echo htmlspecialchars($student['profile_picture']); ?>" alt="Profile" class="student-avatar-large">
                                <?php else: ?>
                                    <div class="student-avatar-large" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; display: flex; align-items: center; justify-content: center; font-size: 48px; font-weight: bold;">
                                        <?php echo strtoupper(substr($student['full_name'], 0, 1)); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="student-info">
                                <h3><?php echo htmlspecialchars($student['full_name']); ?></h3>
                                <p class="student-id">🆔 <?php echo htmlspecialchars($student['id_number'] ?? 'N/A'); ?></p>
                                <p class="student-detail">📧 <?php echo htmlspecialchars($student['email']); ?></p>
                                <p class="student-detail">👤 <?php echo htmlspecialchars($student['username']); ?></p>
                                <p class="student-detail">📚 <?php echo htmlspecialchars($student['year_level'] ?? 'N/A'); ?> - Section <?php echo htmlspecialchars($student['section'] ?? 'N/A'); ?></p>
                                <p class="student-detail"><small>Enrolled: <?php echo date('M d, Y', strtotime($student['enrolled_at'])); ?></small></p>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <p class="empty-message">👥 No students enrolled in this subject yet.</p>
            <?php endif; ?>
        </div>
    </div>

    <script>
        function toggleMenu() {
            const navMenu = document.getElementById('nav-menu');
            navMenu.classList.toggle('active');
        }
    </script>

    <style>
        .info-grid {
            display: grid;
            gap: 20px;
            margin-top: 15px;
        }

        .info-item {
            padding: 15px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 10px;
        }

        .info-item strong {
            color: var(--primary-moss);
            display: block;
            margin-bottom: 8px;
        }

        .students-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .student-card {
            background: var(--card-bg);
            border-radius: 15px;
            padding: 20px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            display: flex;
            flex-direction: column;
            align-items: center;
            text-align: center;
        }

        .student-avatar-container {
            margin-bottom: 15px;
        }

        .student-avatar-large {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            object-fit: cover;
            border: 3px solid var(--primary-moss);
        }

        .student-info {
            width: 100%;
        }

        .student-info h3 {
            margin: 0 0 10px 0;
            color: var(--text-primary);
            font-size: 1.1em;
        }

        .student-id {
            color: var(--primary-moss);
            font-weight: 600;
            margin: 5px 0;
        }

        .student-detail {
            color: var(--text-secondary);
            margin: 5px 0;
            font-size: 0.9em;
        }
    </style>
</body>
</html>
