// Simple C/C++ Syntax Highlighter for Textarea
// Adds color-coded overlay on top of the textarea

class CodeHighlighter {
    constructor(textareaId, overlayId) {
        this.textarea = document.getElementById(textareaId);
        this.overlay = document.getElementById(overlayId);

        if (!this.textarea || !this.overlay) return;

        this.init();
    }

    init() {
        // Sync scroll and input
        this.textarea.addEventListener('input', () => this.highlight());
        this.textarea.addEventListener('scroll', () => this.syncScroll());

        // Initial highlight
        this.highlight();
    }

    syncScroll() {
        this.overlay.scrollTop = this.textarea.scrollTop;
        this.overlay.scrollLeft = this.textarea.scrollLeft;
    }

    highlight() {
        const code = this.textarea.value;
        const highlighted = this.applyHighlighting(code);
        this.overlay.innerHTML = highlighted;
    }

    applyHighlighting(code) {
        // Escape HTML
        code = code.replace(/</g, '&lt;').replace(/>/g, '&gt;');

        // C/C++ Keywords
        const keywords = [
            'auto', 'break', 'case', 'char', 'const', 'continue', 'default', 'do',
            'double', 'else', 'enum', 'extern', 'float', 'for', 'goto', 'if',
            'int', 'long', 'register', 'return', 'short', 'signed', 'sizeof', 'static',
            'struct', 'switch', 'typedef', 'union', 'unsigned', 'void', 'volatile', 'while',
            'class', 'private', 'public', 'protected', 'namespace', 'using', 'new', 'delete',
            'try', 'catch', 'throw', 'template', 'typename', 'virtual', 'friend', 'inline',
            'operator', 'bool', 'true', 'false', 'nullptr', 'this', 'explicit', 'mutable'
        ];

        // Preprocessor directives
        code = code.replace(/(#\s*\w+.*)/g, '<span style="color: #c586c0;">$1</span>');

        // Keywords
        const keywordPattern = new RegExp('\\b(' + keywords.join('|') + ')\\b', 'g');
        code = code.replace(keywordPattern, '<span style="color: #569cd6; font-weight: bold;">$1</span>');

        // Strings
        code = code.replace(/("(?:[^"\\]|\\.)*")/g, '<span style="color: #ce9178;">$1</span>');
        code = code.replace(/('(?:[^'\\]|\\.)*')/g, '<span style="color: #ce9178;">$1</span>');

        // Comments
        code = code.replace(/(\/\/.*)/g, '<span style="color: #6a9955; font-style: italic;">$1</span>');
        code = code.replace(/(\/\*[\s\S]*?\*\/)/g, '<span style="color: #6a9955; font-style: italic;">$1</span>');

        // Numbers
        code = code.replace(/\b(\d+\.?\d*)\b/g, '<span style="color: #b5cea8;">$1</span>');

        // Function calls
        code = code.replace(/\b(\w+)\s*(?=\()/g, '<span style="color: #dcdcaa;">$1</span>');

        // Brackets and operators
        code = code.replace(/([{}[\]()&|+\-*/<>=!%])/g, '<span style="color: #d4d4d4;">$1</span>');

        return code;
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    // Check if elements exist
    const textarea = document.getElementById('code-editor');
    if (textarea) {
        // Apply better default colors
        textarea.style.color = '#9cdcfe';
    }
});
