<?php
session_start();
require_once '../config/database.php';

if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] != 'student') {
    header('Location: ../index.php');
    exit();
}

$student_id = $_SESSION['user_id'];
$conn = getDBConnection();

// Get user data including code points
$user_query = "SELECT profile_picture, code_points FROM users WHERE id = ?";
$stmt = $conn->prepare($user_query);
$stmt->bind_param("i", $student_id);
$stmt->execute();
$user_result = $stmt->get_result();
$user = $user_result->fetch_assoc();
$stmt->close();

// Handle challenge submission
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['submit_answer'])) {
    $challenge_id = intval($_POST['challenge_id']);
    $user_answer = trim($_POST['answer']);

    // Get challenge details
    $stmt = $conn->prepare("SELECT correct_answer, points FROM code_challenges WHERE id = ?");
    $stmt->bind_param("i", $challenge_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $challenge = $result->fetch_assoc();
    $stmt->close();

    if ($challenge) {
        // Check if answer is correct (case-insensitive comparison)
        if (strcasecmp($user_answer, $challenge['correct_answer']) == 0) {
            $today = date('Y-m-d');

            // Check if student hasn't completed this challenge today
            $check_stmt = $conn->prepare("SELECT id FROM challenge_completions WHERE student_id = ? AND challenge_id = ? AND completion_date = ?");
            $check_stmt->bind_param("iis", $student_id, $challenge_id, $today);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();

            if ($check_result->num_rows == 0) {
                // Award points
                $update_points = $conn->prepare("UPDATE users SET code_points = code_points + ? WHERE id = ?");
                $update_points->bind_param("ii", $challenge['points'], $student_id);
                $update_points->execute();
                $update_points->close();

                // Record completion with today's date
                $insert_completion = $conn->prepare("INSERT INTO challenge_completions (student_id, challenge_id, completion_date, is_daily) VALUES (?, ?, ?, TRUE)");
                $insert_completion->bind_param("iis", $student_id, $challenge_id, $today);
                $insert_completion->execute();
                $insert_completion->close();

                $success_message = "Correct! You earned " . $challenge['points'] . " code point(s)!";
            } else {
                $info_message = "You've already completed this challenge today! Come back tomorrow to try again.";
            }
            $check_stmt->close();
        } else {
            $error_message = "Incorrect answer. Try again!";
        }
    }

    // Refresh user data to get updated code points
    $stmt = $conn->prepare("SELECT code_points FROM users WHERE id = ?");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $user_result = $stmt->get_result();
    $user = array_merge($user, $user_result->fetch_assoc());
    $stmt->close();
}

// Get or create today's daily challenge
$today = date('Y-m-d');
$daily_challenge_query = "SELECT challenge_id FROM daily_challenges WHERE challenge_date = ?";
$stmt = $conn->prepare($daily_challenge_query);
$stmt->bind_param("s", $today);
$stmt->execute();
$daily_result = $stmt->get_result();
$daily_challenge_id = null;

if ($daily_result->num_rows == 0) {
    // No daily challenge for today, select a random one
    $stmt->close();
    $random_challenge = $conn->query("SELECT id FROM code_challenges WHERE is_active = 1 ORDER BY RAND() LIMIT 1");
    if ($random_challenge && $random_challenge->num_rows > 0) {
        $daily_challenge_id = $random_challenge->fetch_assoc()['id'];
        // Insert today's daily challenge
        $insert_daily = $conn->prepare("INSERT INTO daily_challenges (challenge_id, challenge_date) VALUES (?, ?)");
        $insert_daily->bind_param("is", $daily_challenge_id, $today);
        $insert_daily->execute();
        $insert_daily->close();
    }
} else {
    $daily_challenge_id = $daily_result->fetch_assoc()['challenge_id'];
    $stmt->close();
}

// Check if student completed today's daily challenge
$daily_completed = false;
if ($daily_challenge_id) {
    $check_daily = $conn->prepare("SELECT id FROM challenge_completions WHERE student_id = ? AND challenge_id = ? AND DATE(completed_at) = ?");
    $check_daily->bind_param("iis", $student_id, $daily_challenge_id, $today);
    $check_daily->execute();
    $daily_completed = $check_daily->get_result()->num_rows > 0;
    $check_daily->close();
}

// Get all challenges with completion status (all challenges reset daily)
$challenges_query = "SELECT c.*,
    CASE WHEN cc_today.id IS NOT NULL THEN 1 ELSE 0 END as completed,
    CASE WHEN cc_today.id IS NOT NULL THEN 1 ELSE 0 END as completed_today,
    CASE WHEN c.id = ? THEN 1 ELSE 0 END as is_daily
    FROM code_challenges c
    LEFT JOIN challenge_completions cc_today ON c.id = cc_today.challenge_id AND cc_today.student_id = ? AND cc_today.completion_date = ?
    WHERE c.is_active = 1
    ORDER BY
        CASE WHEN c.id = ? THEN 0 ELSE 1 END,
        CASE c.difficulty
            WHEN 'easy' THEN 1
            WHEN 'medium' THEN 2
            WHEN 'hard' THEN 3
        END,
        RAND()";
$stmt = $conn->prepare($challenges_query);
$stmt->bind_param("iisi", $daily_challenge_id, $student_id, $today, $daily_challenge_id);
$stmt->execute();
$challenges_result = $stmt->get_result();
$challenges = [];
while ($row = $challenges_result->fetch_assoc()) {
    $challenges[] = $row;
}
$stmt->close();

// Get total available points and earned points
$stats_query = "SELECT
    (SELECT COALESCE(SUM(points), 0) FROM code_challenges WHERE is_active = 1) as total_available,
    (SELECT COALESCE(code_points, 0) FROM users WHERE id = ?) as earned_points,
    (SELECT COUNT(*) FROM challenge_completions WHERE student_id = ?) as completed_count,
    (SELECT COUNT(*) FROM code_challenges WHERE is_active = 1) as total_challenges";
$stmt = $conn->prepare($stats_query);
$stmt->bind_param("ii", $student_id, $student_id);
$stmt->execute();
$stats = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Get unread notifications count
$notif_count_query = "SELECT COUNT(*) as count FROM notifications WHERE user_id = ? AND is_read = FALSE";
$stmt = $conn->prepare($notif_count_query);
$stmt->bind_param("i", $student_id);
$stmt->execute();
$notif_result = $stmt->get_result();
$notif_count = $notif_result->fetch_assoc()['count'];
$stmt->close();

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Code Challenges - CODELAB</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <script src="../assets/js/dark-mode.js" defer>
        // HTML escape function to prevent XSS
        function escapeHtml(text) {
            if (!text) return '';
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
</script>
    <script src="../assets/js/interactive-pet.js" defer></script>
    <script src="../assets/js/push-notifications.js" defer></script>
    <script src="../assets/js/burger-menu.js" defer></script>
</head>
<body>
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <?php if (isset($user['profile_picture']) && $user['profile_picture']): ?>
                    <img src="../<?php echo htmlspecialchars($user['profile_picture']); ?>" alt="Profile" class="nav-profile-pic">
                <?php else: ?>
                    <div class="nav-avatar"><?php echo strtoupper(substr($_SESSION['full_name'], 0, 1)); ?></div>
                <?php endif; ?>
                <h2>CODELAB</h2>
            </div>
            <button class="burger-menu" onclick="toggleMenu()">
                <span></span>
                <span></span>
                <span></span>
            </button>
            <div class="nav-menu" id="nav-menu">
                <a href="dashboard.php" class="nav-link">Dashboard</a>
                <a href="subjects.php" class="nav-link">My Subjects</a>
                <a href="activities.php" class="nav-link">Activities</a>
                <a href="grades.php" class="nav-link">Grades</a>
                <a href="profile.php" class="nav-link">Profile</a>
                <a href="#" class="nav-link notification-bell" onclick="toggleNotifications()">
                    🔔 Notifications
                    <?php if ($notif_count > 0): ?>
                        <span class="badge"><?php echo $notif_count; ?></span>
                    <?php endif; ?>
                </a>
                <a href="../auth/logout.php" class="nav-link logout">Logout</a>
            </div>
        </div>
    </nav>

    <!-- Notifications Panel -->
    <div id="notifications-panel" class="notifications-panel">
        <div class="notifications-header">
            <h3>📬 Notifications</h3>
            <button onclick="toggleNotifications()" class="close-btn">×</button>
        </div>
        <div class="notifications-content" id="notifications-content">
            <p class="loading">Loading notifications...</p>
        </div>
    </div>

    <div class="main-content">
        <div class="dashboard-header">
            <h1>🎮 Code Challenges</h1>
            <p>Test your C/C++ skills and earn code points!</p>
        </div>

        <?php if (isset($success_message)): ?>
            <div class="alert alert-success"><?php echo $success_message; ?></div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="alert alert-error"><?php echo $error_message; ?></div>
        <?php endif; ?>

        <?php if (isset($info_message)): ?>
            <div class="alert alert-info"><?php echo $info_message; ?></div>
        <?php endif; ?>

        <!-- Stats Section -->
        <div class="dashboard-section">
            <h2>📊 Your Progress</h2>
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">🏆</div>
                    <div class="stat-info">
                        <h3>Code Points</h3>
                        <p class="stat-value"><?php echo $stats['earned_points']; ?></p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">✅</div>
                    <div class="stat-info">
                        <h3>Completed</h3>
                        <p class="stat-value"><?php echo $stats['completed_count']; ?> / <?php echo $stats['total_challenges']; ?></p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">🎯</div>
                    <div class="stat-info">
                        <h3>Progress</h3>
                        <p class="stat-value"><?php echo $stats['total_challenges'] > 0 ? round(($stats['completed_count'] / $stats['total_challenges']) * 100) : 0; ?>%</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Challenges Section -->
        <div class="dashboard-section">
            <h2>💻 Challenges</h2>

            <?php if ($daily_challenge_id && !$daily_completed): ?>
                <div class="alert" style="background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%); color: #000; border: none; margin-bottom: 20px; padding: 15px; border-radius: 12px; box-shadow: 0 4px 15px rgba(255, 215, 0, 0.3); animation: dailyGlow 2s ease-in-out infinite;">
                    <strong>⭐ Daily Challenge Available!</strong> Complete today's challenge to earn bonus points! Resets every 24 hours.
                </div>
            <?php elseif ($daily_completed): ?>
                <div class="alert alert-success" style="margin-bottom: 20px;">
                    <strong>✅ Daily Challenge Completed!</strong> Come back tomorrow for a new challenge!
                </div>
            <?php endif; ?>

            <div class="challenges-grid">
                <?php foreach ($challenges as $challenge): ?>
                    <div class="challenge-card-preview <?php echo $challenge['completed'] ? 'completed' : ''; ?> <?php echo $challenge['is_daily'] ? 'daily-challenge' : ''; ?>" data-difficulty="<?php echo $challenge['difficulty']; ?>" <?php echo $challenge['is_daily'] ? 'style="border: 3px solid #FFD700; box-shadow: 0 8px 30px rgba(255, 215, 0, 0.4); position: relative; overflow: visible;"' : ''; ?>>
                        <?php if ($challenge['is_daily']): ?>
                            <div style="position: absolute; top: -15px; left: 50%; transform: translateX(-50%); background: linear-gradient(135deg, #FFD700, #FFA500); color: #000; padding: 8px 20px; border-radius: 20px; font-weight: bold; font-size: 0.9em; box-shadow: 0 4px 10px rgba(255, 215, 0, 0.5); z-index: 10; white-space: nowrap;">
                                ⭐ TODAY'S DAILY CHALLENGE ⭐
                            </div>
                        <?php endif; ?>

                        <div class="challenge-header" style="<?php echo $challenge['is_daily'] ? 'margin-top: 20px;' : ''; ?>">
                            <h3><?php echo htmlspecialchars($challenge['title']); ?></h3>
                            <div class="challenge-badges">
                                <span class="difficulty-badge <?php echo $challenge['difficulty']; ?>">
                                    <?php echo ucfirst($challenge['difficulty']); ?>
                                </span>
                                <span class="points-badge"><?php echo $challenge['points']; ?> pts</span>
                                <?php if ($challenge['completed']): ?>
                                    <span class="completed-badge">✓ Completed Today</span>
                                <?php endif; ?>
                            </div>
                        </div>

                        <p class="challenge-description"><?php echo htmlspecialchars($challenge['description']); ?></p>

                        <div class="code-display-preview">
                            <pre><code><?php echo htmlspecialchars(substr($challenge['code_template'], 0, 100)); ?>...</code></pre>
                        </div>

                        <?php if (!$challenge['completed']): ?>
                            <button type="button" class="btn btn-primary btn-block" onclick="startChallenge(<?php echo $challenge['id']; ?>)">
                                🚀 Start Challenge
                            </button>
                        <?php else: ?>
                            <div class="completed-message">
                                ✓ You've completed this challenge today and earned <?php echo $challenge['points']; ?> point(s)! Come back tomorrow for more challenges.
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

    <!-- Fullscreen Modals (Outside grid for proper z-index) -->
    <?php foreach ($challenges as $challenge): ?>
        <div class="challenge-fullscreen" id="challenge-fullscreen-<?php echo $challenge['id']; ?>">
            <div class="fullscreen-container">
                <div class="fullscreen-header">
                    <h2><?php echo htmlspecialchars($challenge['title']); ?></h2>
                    <button class="close-fullscreen" onclick="closeChallenge(<?php echo $challenge['id']; ?>)">✕ Exit</button>
                </div>

                <div class="fullscreen-badges">
                    <span class="difficulty-badge <?php echo $challenge['difficulty']; ?>">
                        <?php echo ucfirst($challenge['difficulty']); ?>
                    </span>
                    <span class="points-badge"><?php echo $challenge['points']; ?> pts</span>
                </div>

                <p class="fullscreen-description"><?php echo htmlspecialchars($challenge['description']); ?></p>

                <div class="code-display-fullscreen">
                    <table class="code-table">
                        <?php
                        $code_formatted = str_replace('\n', "\n", $challenge['code_template']);
                        $code_lines = explode("\n", $code_formatted);
                        foreach ($code_lines as $line_num => $line_code):
                        ?>
                        <tr>
                            <td class="line-number"><?php echo $line_num + 1; ?></td>
                            <td class="line-code"><pre><?php echo htmlspecialchars($line_code); ?></pre></td>
                        </tr>
                        <?php endforeach; ?>
                    </table>
                </div>

                <form method="POST" class="challenge-form">
                    <input type="hidden" name="challenge_id" value="<?php echo $challenge['id']; ?>">
                    <div class="form-group">
                        <label>Your Answer (fill in the blank):</label>
                        <input type="text" name="answer" class="answer-input" required placeholder="Type your answer here..." autocomplete="off">
                    </div>
                    <div class="challenge-actions">
                        <button type="button" class="btn btn-secondary" onclick="showHint(<?php echo $challenge['id']; ?>)">💡 Hint</button>
                        <button type="submit" name="submit_answer" class="btn btn-primary">Submit Answer</button>
                    </div>
                    <div class="hint-box" id="hint-<?php echo $challenge['id']; ?>" style="display: none;">
                        <strong>Hint:</strong> <?php echo htmlspecialchars($challenge['hint']); ?>
                    </div>
                </form>
            </div>
        </div>
    <?php endforeach; ?>

    <script>
        function showHint(challengeId) {
            const hintBox = document.getElementById('hint-' + challengeId);
            hintBox.style.display = hintBox.style.display === 'none' ? 'block' : 'none';
        }

        function startChallenge(challengeId) {
            const fullscreen = document.getElementById('challenge-fullscreen-' + challengeId);
            fullscreen.classList.add('active');
            document.body.style.overflow = 'hidden';

            // Focus on the answer input
            setTimeout(() => {
                const input = fullscreen.querySelector('.answer-input');
                if (input) input.focus();
            }, 100);
        }

        function closeChallenge(challengeId) {
            const fullscreen = document.getElementById('challenge-fullscreen-' + challengeId);
            fullscreen.classList.remove('active');
            document.body.style.overflow = '';
        }

        // ESC key to exit fullscreen
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                const activeFullscreen = document.querySelector('.challenge-fullscreen.active');
                if (activeFullscreen) {
                    const challengeId = activeFullscreen.id.replace('challenge-fullscreen-', '');
                    closeChallenge(challengeId);
                }
            }
        });
    </script>

    <script>
function toggleNotifications() {
    const panel = document.getElementById('notifications-panel');
    const isOpen = panel.classList.contains('open');

    if (isOpen) {
        panel.classList.remove('open');
    } else {
        panel.classList.add('open');
        loadNotifications();
    }
}

function loadNotifications() {
    const content = document.getElementById('notifications-content');

    fetch('get_notifications.php')
        .then(response => response.json())
        .then(data => {
            if (data.length === 0) {
                content.innerHTML = '<p class="empty-message">No notifications yet.</p>';
            } else {
                let html = '';
                data.forEach(notif => {
                    html += `
                        <div class="notification-item ${notif.is_read ? '' : 'unread'}">
                            <p>${escapeHtml(notif.message)}</p>
                            <span class="notif-time">${escapeHtml(notif.created_at)}</span>
                            ${notif.activity_id ? `<a href="take_activity.php?id=${notif.activity_id}" class="notif-link">View Activity →</a>` : ''}
                        </div>
                    `;
                });
                content.innerHTML = html;

                // Mark notifications as read
                fetch('mark_notifications_read.php');
            }
        })
        .catch(error => {
            content.innerHTML = '<p class="error-message">Failed to load notifications.</p>';
        });
}

// Close notifications panel when clicking outside
document.addEventListener('click', function(event) {
    const panel = document.getElementById('notifications-panel');
    const target = event.target;

    if (!panel.contains(target) && !target.closest('.nav-link')) {
        panel.classList.remove('open');
    }
});
    </script>

    <style>
        .challenges-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(500px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .challenge-card-preview {
            background: var(--bg-card);
            border: 2px solid var(--border-color);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
            position: relative;
        }

        .challenge-card-preview:hover {
            transform: translateY(-3px);
            box-shadow: 0 5px 20px var(--shadow-hover);
        }

        .challenge-card-preview.completed {
            border-color: var(--btn-secondary);
            background: var(--bg-light);
        }

        .challenge-card-preview.daily-challenge {
            background: linear-gradient(135deg, rgba(255, 215, 0, 0.05), rgba(255, 165, 0, 0.05));
        }

        .code-display-preview {
            background: var(--code-bg);
            border-radius: 8px;
            padding: 15px;
            margin: 15px 0;
            overflow-x: auto;
            max-height: 100px;
            overflow-y: hidden;
        }

        .btn-block {
            width: 100%;
            margin-top: 15px;
        }

        /* Fullscreen Challenge Styles */
        .challenge-fullscreen {
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100vw !important;
            height: 100vh !important;
            background: rgba(0, 0, 0, 0.95) !important;
            z-index: 999999 !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            padding: 20px !important;
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s ease, visibility 0.3s ease;
            pointer-events: none;
        }

        .challenge-fullscreen.active {
            opacity: 1 !important;
            visibility: visible !important;
            pointer-events: auto !important;
        }

        .fullscreen-container {
            background: var(--bg-card) !important;
            border-radius: 16px;
            padding: 30px;
            max-width: 1200px;
            width: 100%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: 0 10px 50px rgba(0, 0, 0, 0.5);
            transform: scale(0.95);
            transition: transform 0.3s ease;
            position: relative;
            z-index: 1000000;
        }

        .challenge-fullscreen.active .fullscreen-container {
            transform: scale(1);
        }

        .fullscreen-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid var(--border-color);
        }

        .fullscreen-header h2 {
            color: var(--text-primary);
            margin: 0;
        }

        .close-fullscreen {
            background: #dc3545;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .close-fullscreen:hover {
            background: #c82333;
            transform: scale(1.05);
        }

        .fullscreen-badges {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }

        .fullscreen-description {
            color: var(--text-secondary);
            font-size: 1.1em;
            margin-bottom: 20px;
            line-height: 1.6;
        }

        @keyframes dailyGlow {
            0%, 100% {
                box-shadow: 0 4px 15px rgba(255, 215, 0, 0.3);
            }
            50% {
                box-shadow: 0 4px 25px rgba(255, 215, 0, 0.6);
            }
        }

        .challenge-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
            flex-wrap: wrap;
            gap: 10px;
        }

        .challenge-header h3 {
            color: var(--text-primary);
            margin: 0;
        }

        .challenge-badges {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }

        .difficulty-badge, .points-badge, .completed-badge {
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 0.85em;
            font-weight: 600;
        }

        .difficulty-badge.easy {
            background: #d4edda;
            color: #155724;
        }

        .difficulty-badge.medium {
            background: #fff3cd;
            color: #856404;
        }

        .difficulty-badge.hard {
            background: #f8d7da;
            color: #721c24;
        }

        .points-badge {
            background: var(--accent-color);
            color: var(--text-on-primary);
        }

        .completed-badge {
            background: var(--btn-secondary);
            color: var(--text-on-primary);
        }

        .challenge-description {
            color: var(--text-secondary);
            margin-bottom: 15px;
        }

        .code-display {
            background: var(--code-bg);
            border-radius: 8px;
            padding: 15px;
            margin: 15px 0;
            overflow-x: auto;
        }

        .code-display pre {
            margin: 0;
            white-space: pre-wrap;
            word-wrap: break-word;
        }

        .code-display code {
            color: var(--code-text);
            font-family: 'Courier New', monospace;
            font-size: 1em;
            line-height: 1.6;
            white-space: pre;
            display: block;
        }

        /* Fullscreen Code Display with Line Numbers */
        .code-display-fullscreen {
            background: #1e1e1e;
            border-radius: 8px;
            margin: 20px 0;
            overflow-x: auto;
            border: 1px solid #3e3e3e;
        }

        .code-table {
            width: 100%;
            border-collapse: collapse;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }

        .code-table tr {
            background: #1e1e1e;
        }

        .line-number {
            background: #252526;
            color: #858585;
            text-align: right;
            padding: 0 15px;
            user-select: none;
            vertical-align: top;
            width: 50px;
            border-right: 1px solid #3e3e3e;
        }

        .line-code {
            color: #d4d4d4;
            padding: 0 15px;
            vertical-align: top;
        }

        .line-code pre {
            margin: 0;
            padding: 0;
            white-space: pre;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }

        .answer-input {
            width: 100%;
            padding: 12px;
            border: 2px solid var(--input-border);
            border-radius: 8px;
            font-size: 1em;
            font-family: 'Courier New', monospace;
            background: var(--bg-card);
            color: var(--text-primary);
        }

        .answer-input:focus {
            outline: none;
            border-color: var(--input-focus);
        }

        .challenge-actions {
            display: flex;
            gap: 10px;
            margin-top: 15px;
        }

        .hint-box {
            margin-top: 15px;
            padding: 12px;
            background: var(--bg-light);
            border-left: 4px solid var(--accent-color);
            border-radius: 4px;
            color: var(--text-primary);
        }

        .completed-message {
            text-align: center;
            padding: 15px;
            background: var(--bg-light);
            border-radius: 8px;
            color: var(--btn-secondary);
            font-weight: 600;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .alert-info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }

        @media (max-width: 768px) {
            .challenges-grid {
                grid-template-columns: 1fr;
            }

            .challenge-header {
                flex-direction: column;
            }

            .challenge-actions {
                flex-direction: column;
            }

            .fullscreen-container {
                padding: 20px;
            }
        }
    </style>
</body>
</html>
