<?php
/**
 * Setup Test Script
 * Run this after deployment to verify everything is working
 * Access: https://yourdomain.com/test_setup.php
 * DELETE THIS FILE AFTER SUCCESSFUL TESTING!
 */

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Setup Test - C++ Learning System</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            max-width: 900px;
            margin: 40px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            border-bottom: 3px solid #3498db;
            padding-bottom: 10px;
        }
        h2 {
            color: #34495e;
            margin-top: 30px;
        }
        .test-item {
            padding: 15px;
            margin: 10px 0;
            border-radius: 5px;
            border-left: 4px solid #ccc;
        }
        .success {
            background: #d4edda;
            border-color: #28a745;
            color: #155724;
        }
        .warning {
            background: #fff3cd;
            border-color: #ffc107;
            color: #856404;
        }
        .error {
            background: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }
        .info {
            background: #d1ecf1;
            border-color: #17a2b8;
            color: #0c5460;
        }
        .badge {
            padding: 3px 8px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 12px;
            margin-left: 10px;
        }
        .badge-success { background: #28a745; color: white; }
        .badge-error { background: #dc3545; color: white; }
        .badge-warning { background: #ffc107; color: black; }
        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
        }
        .warning-box {
            background: #fff3cd;
            border: 2px solid #ffc107;
            padding: 20px;
            border-radius: 5px;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Setup Test Results</h1>
        <div class="warning-box">
            <strong>⚠️ IMPORTANT:</strong> Delete this file (<code>test_setup.php</code>) after testing!
        </div>

        <?php
        $all_tests_passed = true;

        // Test 1: PHP Version
        echo '<h2>1. PHP Version</h2>';
        $php_version = phpversion();
        if (version_compare($php_version, '7.4.0', '>=')) {
            echo '<div class="test-item success">✅ PHP Version: ' . $php_version . ' <span class="badge badge-success">PASS</span></div>';
        } else {
            echo '<div class="test-item error">❌ PHP Version: ' . $php_version . ' (Required: 7.4+) <span class="badge badge-error">FAIL</span></div>';
            $all_tests_passed = false;
        }

        // Test 2: Required Extensions
        echo '<h2>2. Required PHP Extensions</h2>';
        $required_extensions = ['mysqli', 'curl', 'json', 'session'];
        foreach ($required_extensions as $ext) {
            if (extension_loaded($ext)) {
                echo '<div class="test-item success">✅ ' . $ext . ' extension <span class="badge badge-success">LOADED</span></div>';
            } else {
                echo '<div class="test-item error">❌ ' . $ext . ' extension <span class="badge badge-error">MISSING</span></div>';
                $all_tests_passed = false;
            }
        }

        // Test 3: .env File
        echo '<h2>3. Configuration File</h2>';
        $env_file = __DIR__ . '/.env';
        if (file_exists($env_file)) {
            echo '<div class="test-item success">✅ .env file exists <span class="badge badge-success">FOUND</span></div>';

            // Check if .env is configured
            $env_content = file_get_contents($env_file);
            if (strpos($env_content, 'your_cpanel_db_user') !== false ||
                strpos($env_content, 'your_secure_password') !== false ||
                strpos($env_content, 'CHANGE_THIS') !== false) {
                echo '<div class="test-item warning">⚠️ .env file needs configuration <span class="badge badge-warning">ACTION REQUIRED</span></div>';
                echo '<div class="test-item info">📝 Update database credentials and access codes in .env file</div>';
            } else {
                echo '<div class="test-item success">✅ .env file appears to be configured <span class="badge badge-success">CONFIGURED</span></div>';
            }
        } else {
            echo '<div class="test-item error">❌ .env file not found <span class="badge badge-error">MISSING</span></div>';
            $all_tests_passed = false;
        }

        // Test 4: Directory Permissions
        echo '<h2>4. Directory Permissions</h2>';
        $directories = ['cache', 'uploads'];
        foreach ($directories as $dir) {
            $dir_path = __DIR__ . '/' . $dir;
            if (is_dir($dir_path)) {
                if (is_writable($dir_path)) {
                    echo '<div class="test-item success">✅ ' . $dir . '/ is writable <span class="badge badge-success">OK</span></div>';
                } else {
                    echo '<div class="test-item error">❌ ' . $dir . '/ is not writable (chmod to 755 or 777) <span class="badge badge-error">FIX NEEDED</span></div>';
                    $all_tests_passed = false;
                }
            } else {
                echo '<div class="test-item warning">⚠️ ' . $dir . '/ directory not found (will be created automatically) <span class="badge badge-warning">MISSING</span></div>';
            }
        }

        // Test 5: Database Connection
        echo '<h2>5. Database Connection</h2>';
        require_once __DIR__ . '/config/database.php';
        try {
            $conn = getDBConnection();
            if ($conn) {
                echo '<div class="test-item success">✅ Database connection successful <span class="badge badge-success">CONNECTED</span></div>';
                echo '<div class="test-item info">📊 Database: ' . DB_NAME . '</div>';
                $conn->close();
            }
        } catch (Exception $e) {
            echo '<div class="test-item error">❌ Database connection failed <span class="badge badge-error">FAILED</span></div>';
            echo '<div class="test-item error">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            $all_tests_passed = false;
        }

        // Test 6: Piston API Connectivity
        echo '<h2>6. Code Execution API</h2>';
        if (function_exists('curl_init')) {
            $ch = curl_init('https://emkc.org/api/v2/piston/runtimes');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($http_code === 200) {
                echo '<div class="test-item success">✅ Piston API is accessible <span class="badge badge-success">ONLINE</span></div>';
                echo '<div class="test-item info">🚀 Code execution will work correctly</div>';
            } else {
                echo '<div class="test-item warning">⚠️ Cannot reach Piston API (HTTP ' . $http_code . ') <span class="badge badge-warning">CHECK CONNECTION</span></div>';
                echo '<div class="test-item info">🌐 Check your server\'s internet connectivity</div>';
            }
        }

        // Test 7: Session Support
        echo '<h2>7. Session Support</h2>';
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $_SESSION['test'] = 'working';
        if (isset($_SESSION['test']) && $_SESSION['test'] === 'working') {
            echo '<div class="test-item success">✅ PHP sessions are working <span class="badge badge-success">OK</span></div>';
            unset($_SESSION['test']);
        } else {
            echo '<div class="test-item error">❌ PHP sessions not working <span class="badge badge-error">FAILED</span></div>';
            $all_tests_passed = false;
        }

        // Final Summary
        echo '<h2>Summary</h2>';
        if ($all_tests_passed) {
            echo '<div class="test-item success" style="font-size: 18px; font-weight: bold;">';
            echo '✅ All critical tests passed! Your system is ready to use.';
            echo '</div>';
            echo '<div class="test-item info">';
            echo '📝 <strong>Next Steps:</strong><br>';
            echo '1. Delete this test file: <code>test_setup.php</code><br>';
            echo '2. Visit your homepage to start using the system<br>';
            echo '3. Register a test account to verify functionality<br>';
            echo '4. Test code execution with a simple program';
            echo '</div>';
        } else {
            echo '<div class="test-item error" style="font-size: 18px; font-weight: bold;">';
            echo '❌ Some tests failed. Please fix the issues above before proceeding.';
            echo '</div>';
        }
        ?>

        <div class="test-item info" style="margin-top: 30px;">
            <strong>📚 Need Help?</strong><br>
            Refer to <code>DEPLOYMENT_GUIDE.md</code> for detailed instructions.
        </div>

        <div class="warning-box" style="margin-top: 30px;">
            <strong>🔒 SECURITY REMINDER:</strong><br>
            • Delete this file after testing!<br>
            • Update .env with strong passwords<br>
            • Change access codes<br>
            • Enable HTTPS
        </div>
    </div>
</body>
</html>
